// content.js
if (!window.listingGuardScriptLoaded) {
  window.listingGuardScriptLoaded = true;
  console.log('ListingGuard AI Content Script Loaded');

  // --- 辅助函数：高亮抓取到的元素 ---
  function highlightElement(element) {
    if (!element) return;
    
    // 保存原始样式
    const originalTransition = element.style.transition;
    const originalBoxShadow = element.style.boxShadow;
    const originalBackgroundColor = element.style.backgroundColor;

    // 应用高亮样式 (蓝色边框 + 淡黄色背景)
    element.style.transition = 'all 0.5s ease';
    element.style.boxShadow = '0 0 0 4px rgba(59, 130, 246, 0.5)'; // 蓝色光晕
    element.style.backgroundColor = 'rgba(254, 240, 138, 0.3)'; // 淡黄色背景

    // 3秒后恢复
    setTimeout(() => {
      element.style.boxShadow = originalBoxShadow;
      element.style.backgroundColor = originalBackgroundColor;
      // 稍微延迟一点移除 transition，让恢复过程也有动画
      setTimeout(() => {
        element.style.transition = originalTransition;
      }, 500);
    }, 3000);
  }

  // --- 核心抓取逻辑 ---
  function scrapeAmazonData() {
    const data = {
      title: '',
      bullets: [],
      description: '',
      fallback: []
    };

    // 1. 抓取标题 (Title)
    // 优先级: id="item_name" -> name="item_name" -> [data-automation-id="title-textarea"]
    const titleSelectors = [
      '#item_name', 
      '[name="item_name"]', 
      '[data-automation-id="title-textarea"]'
    ];
    
    for (const selector of titleSelectors) {
      const el = document.querySelector(selector);
      if (el && el.value && el.value.trim()) {
        data.title = el.value.trim();
        highlightElement(el);
        console.log(`Title found via ${selector}`);
        break; // 找到即止
      }
    }

    // 2. 抓取五点描述 (Bullet Points)
    // 循环查找 id 或 name 包含 bullet_point_ 从 1 到 10
    for (let i = 0; i < 10; i++) { // 通常是5个，但多找几个以防万一
      // 模糊匹配: id 或 name 包含 bullet_point_X (注意: Amazon 有时是 bullet_point_1, 有时是 bullet-point-1, 这里主要匹配 key string)
      // 根据指令明确要求: id 或 name 包含 bullet_point_
      // 我们构建一个更广泛的查找策略
      
      // 尝试直接构建可能的 ID/Name
      // 常见模式: bullet_point_1, bullet-point-1, feature_bullet_1
      // 指令特指: "包含 bullet_point_ 从 1 到 10"
      
      // 使用 querySelectorAll 查找所有输入框，然后正则匹配
      // 为了性能，我们先尝试直接查找
      // 实际上 Amazon 常用的是 #bullet_point_1, #bullet_point_2 ...
      
      // 让我们遵循"循环查找"的指令，结合 querySelector
      // 尝试 ID
      let el = document.getElementById(`bullet_point_${i+1}`) || 
               document.getElementsByName(`bullet_point_${i+1}`)[0];
      
      if (el && el.value && el.value.trim()) {
        data.bullets.push(el.value.trim());
        highlightElement(el);
      }
    }
    
    // 如果上面的精确查找没找到，尝试更宽泛的搜索 (针对变体)
    if (data.bullets.length === 0) {
       const inputs = document.querySelectorAll('input[type="text"], textarea');
       inputs.forEach(input => {
           const id = input.id || '';
           const name = input.name || '';
           if ((id.includes('bullet_point') || name.includes('bullet_point')) && input.value.trim()) {
               // 避免重复 (如果上面逻辑有漏网之鱼)
               if (!data.bullets.includes(input.value.trim())) {
                   data.bullets.push(input.value.trim());
                   highlightElement(input);
               }
           }
       });
    }


    // 3. 抓取产品描述 (Product Description)
    // 优先级: id="product_description" -> name="product_description" -> 页面最大 textarea
    const descSelectors = ['#product_description', '[name="product_description"]'];
    let descFound = false;
    
    for (const selector of descSelectors) {
      const el = document.querySelector(selector);
      if (el && el.value && el.value.trim()) {
        data.description = el.value.trim();
        highlightElement(el);
        descFound = true;
        break;
      }
    }

    // 如果没找到，找最大的 textarea
    if (!descFound) {
      let maxLen = 0;
      let maxEl = null;
      document.querySelectorAll('textarea').forEach(el => {
        // 排除已知的 title 或 bullets (虽然 bullets 通常是 input，但有时也是 textarea)
        // 简单判断: 长度最长且可见
        if (el.offsetParent !== null && el.value.length > maxLen) {
          maxLen = el.value.length;
          maxEl = el;
        }
      });
      
      // 只有当长度足够长才认为是描述 (例如 > 50 字符)
      if (maxEl && maxLen > 50) {
          data.description = maxEl.value.trim();
          highlightElement(maxEl);
          console.log('Description found via Largest Textarea');
          descFound = true;
      }
    }

    // 4. 智能兜底 (Smart Fallback)
    // 如果上述特定 ID 都没找到
    const hasSpecificContent = data.title || data.bullets.length > 0 || data.description;
    
    if (!hasSpecificContent) {
      console.log('未找到特定目标，启动智能兜底...');
      const allTextareas = Array.from(document.querySelectorAll('textarea'))
        .filter(el => {
          const val = el.value.trim();
          return el.offsetParent !== null && val.length > 20; // 可见且字数 > 20
        });
        
      allTextareas.forEach(el => {
          data.fallback.push(el.value.trim());
          highlightElement(el);
      });
    }

    return data;
  }

  chrome.runtime.onMessage.addListener((request, sender, sendResponse) => {
    if (request.action === 'extract_text') {
      try {
          const data = scrapeAmazonData();
          
          // 检查是否抓取成功
          const success = data.title || data.bullets.length > 0 || data.description || data.fallback.length > 0;
          
          sendResponse({ 
            success: success,
            data: data 
          });
      } catch (e) {
          console.error('Scraping error:', e);
          sendResponse({ success: false, error: e.message });
      }
    }
    return true; // 保持消息通道开启
  });

}
