// --- Config ---
const SUPABASE_URL = 'https://wgdstfxlqipybdhfiqtg.supabase.co';
const SUPABASE_ANON_KEY = 'sb_publishable_PMv5DIt8zpfoR25lyW19pw_kMVo5Gew';

// --- Init Supabase ---
let supabaseClient;
try {
    if (window.supabase) {
        supabaseClient = window.supabase.createClient(SUPABASE_URL, SUPABASE_ANON_KEY, {
            auth: {
                persistSession: true,
                storage: window.localStorage,
                autoRefreshToken: true,
                detectSessionInUrl: false
            }
        });
        console.log('Supabase client initialized');
    } else {
        console.error('Supabase SDK not loaded');
        // Wait for DOM
        document.addEventListener('DOMContentLoaded', () => {
            const authMsg = document.getElementById('auth-msg');
            if (authMsg) authMsg.innerText = 'Error: Supabase SDK failed to load';
        });
    }
} catch (e) {
    console.error('Supabase init failed:', e);
}

// --- Core Logic ---
document.addEventListener('DOMContentLoaded', () => {
    // 1. Env Detection
    const isExtension = window.chrome && chrome.runtime && chrome.runtime.id;
    if (isExtension) {
        document.body.classList.add('is-extension');
    } else {
        document.body.classList.add('is-web');
    }
    // --- DOM Elements ---
    const els = {
        landingView: document.getElementById('landing-view'),
        appView: document.getElementById('app-view'),
        navPublic: document.getElementById('nav-public'),
        navPrivate: document.getElementById('nav-private'),
        creditCount: document.getElementById('credit-count'),
        emailInput: document.getElementById('email-input'),
        pwdInput: document.getElementById('password-input'),
        authMsg: document.getElementById('auth-msg'),
        productInput: document.getElementById('product-input'),
        platformSelect: document.getElementById('platform-select'),
        checkBtn: document.getElementById('check-btn'),
        logoutBtn: document.getElementById('logout-btn'),
        navLoginBtn: document.getElementById('nav-login-btn'),
        copyBtn: document.getElementById('copy-btn'),
        loginSubmitBtn: document.getElementById('login-submit-btn'),
        signupSubmitBtn: document.getElementById('signup-submit-btn'),
        autoFillBtn: document.getElementById('auto-fill-btn'),
        
        // New Buttons
        startTrialBtn: document.getElementById('start-trial-btn'),
        signupScrollBtn: document.getElementById('signup-scroll-btn'),
        waitlistBtns: document.querySelectorAll('.waitlist-btn'),
        
        // Result Area
        resultPlaceholder: document.getElementById('result-placeholder'),
        resultLoading: document.getElementById('result-loading'),
        resultContent: document.getElementById('result-content'),
        scoreVal: document.getElementById('score-val'),
        scoreBadge: document.getElementById('score-badge'),
        summaryText: document.getElementById('summary-text'),
        detailsList: document.getElementById('details-list'),
        safeCopyText: document.getElementById('safe-copy-text'),
        
        // Get More Button
        getMoreBtn: document.getElementById('get-more-btn'),

        // Loading
        loadingBar: document.getElementById('loading-bar'),
        loadingPercent: document.getElementById('loading-percent'),
        loadingStep: document.getElementById('loading-step'),

        // Modal Elements
        installBtn: document.getElementById('install-btn'),
        downloadExtBtn: document.getElementById('download-ext-btn'),
        installModal: document.getElementById('install-modal'),
        closeModal: document.getElementById('close-modal'),
        
        // Waitlist Modal
        waitlistModal: document.getElementById('waitlist-modal'),
        closeWaitlist: document.getElementById('close-waitlist'),
        
        // Credits Modal
        creditsModal: document.getElementById('credits-modal'),
        closeCreditsModal: document.getElementById('close-credits-modal'),
        goPricingBtn: document.getElementById('go-pricing-btn'),

        // Logo
        homeLogo: document.getElementById('home-logo')
    };

    // 1. Init & Listeners
    async function init() {
        if (!supabaseClient) return;

        // Logo Click
        if (els.homeLogo) {
            els.homeLogo.addEventListener('click', () => {
                const isExt = document.body.classList.contains('is-extension');
                if (isExt) {
                    chrome.tabs.create({ url: 'https://www.listingguardai.com' });
                } else {
                    window.location.href = 'index.html';
                }
            });
        }
        
        // Credits Modal Logic
        if (els.creditsModal) {
            if (els.closeCreditsModal) {
                els.closeCreditsModal.addEventListener('click', () => {
                    els.creditsModal.classList.add('hidden');
                });
            }
            if (els.goPricingBtn) {
                els.goPricingBtn.addEventListener('click', () => {
                    const isExt = document.body.classList.contains('is-extension');
                    const url = 'https://www.listingguardai.com/pricing.html';
                    if (isExt) {
                        chrome.tabs.create({ url });
                    } else {
                        window.open(url, '_blank');
                    }
                    els.creditsModal.classList.add('hidden');
                });
            }
            window.addEventListener('click', (e) => {
                if (e.target === els.creditsModal) {
                    els.creditsModal.classList.add('hidden');
                }
            });
        }

        // --- Event Listeners for New Buttons ---
        
        // Scroll to Auth
        const scrollToAuth = () => {
            const authSection = document.getElementById('auth-section-web') || document.getElementById('auth-section');
            if (authSection) {
                authSection.scrollIntoView({ behavior: 'smooth' });
            }
        };
        
        if (els.startTrialBtn) els.startTrialBtn.addEventListener('click', scrollToAuth);
        if (els.signupScrollBtn) els.signupScrollBtn.addEventListener('click', scrollToAuth);
        
        // Waitlist Modal
        if (els.waitlistModal) {
            els.waitlistBtns.forEach(btn => {
                btn.addEventListener('click', () => {
                    els.waitlistModal.classList.remove('hidden');
                });
            });
            
            if (els.closeWaitlist) {
                els.closeWaitlist.addEventListener('click', () => {
                    els.waitlistModal.classList.add('hidden');
                });
            }
            
            window.addEventListener('click', (e) => {
                if (e.target === els.waitlistModal) {
                    els.waitlistModal.classList.add('hidden');
                }
            });
        }

        // Download Modal Logic
        const showDownloadModal = (e) => {
            if (e) e.preventDefault();
            if (els.installModal) els.installModal.classList.remove('hidden');
        };

        if (els.installBtn) els.installBtn.addEventListener('click', showDownloadModal);
        if (els.downloadExtBtn) els.downloadExtBtn.addEventListener('click', showDownloadModal);

        if (els.installModal) {
            if (els.closeModal) {
                els.closeModal.addEventListener('click', () => {
                    els.installModal.classList.add('hidden');
                });
            }

            // Close on click outside
            window.addEventListener('click', (e) => {
                if (e.target === els.installModal) {
                    els.installModal.classList.add('hidden');
                }
            });
        }

        // Auth State Listener
        supabaseClient.auth.onAuthStateChange(async (event, session) => {
            console.log('Auth event:', event);
            
            if (event === 'SIGNED_OUT' || event === 'USER_DELETED') {
                handleAuthStateChange(null);
            } else if (event === 'SIGNED_IN' || event === 'TOKEN_REFRESHED') {
                handleAuthStateChange(session);
            }
        });

        // Check Initial Session
        try {
            const { data, error } = await supabaseClient.auth.getSession();
            
            if (error) {
                throw error;
            }
            
            handleAuthStateChange(data.session);
        } catch (error) {
            console.warn('Session init error:', error);
            // Handle AuthApiError or Invalid Refresh Token
            if (error.message && (
                error.message.includes('Refresh Token Not Found') || 
                error.message.includes('Invalid Refresh Token') ||
                error.message.includes('invalid_grant') ||
                error.name === 'AuthApiError'
            )) {
                console.log('Critical auth error, forcing cleanup...');
                await supabaseClient.auth.signOut();
                localStorage.clear(); // Clear local storage to be safe
                handleAuthStateChange(null);
            }
        }
    }

    // 2. State Handler
    async function handleAuthStateChange(session) {
        const isLt = !!session;
        const isExt = document.body.classList.contains('is-extension');
        
        if (isLt) {
            // --- Logged In ---
            els.landingView.classList.add('hidden');
            els.appView.classList.remove('hidden');
            els.navPublic.classList.add('hidden');
            els.navPrivate.classList.remove('hidden');
            
            // Auto Grab Button Visibility
            if (els.autoFillBtn) {
                if (isExt) {
                    els.autoFillBtn.classList.remove('hidden');
                } else {
                    els.autoFillBtn.classList.add('hidden');
                }
            }
            
            await fetchCredits(session.user.id);
        } else {
            // --- Logged Out ---
            els.landingView.classList.remove('hidden');
            els.appView.classList.add('hidden');
            els.navPublic.classList.remove('hidden');
            els.navPrivate.classList.add('hidden');
            
            // Landing Page Adapt
            const extLanding = document.querySelector('.extension-landing');
            const webLanding = document.querySelector('.web-landing');
            
            if (isExt) {
                if (extLanding) extLanding.classList.remove('hidden');
                if (webLanding) webLanding.classList.add('hidden');
            } else {
                if (extLanding) extLanding.classList.add('hidden');
                if (webLanding) webLanding.classList.remove('hidden');
            }

            els.creditCount.innerText = '-';
        }
    }

    // 3. Fetch Credits
    async function fetchCredits(userId) {
        if (!userId) return;
        try {
            const { data, error } = await supabaseClient
                .from('profiles')
                .select('credits')
                .eq('id', userId)
                .single();

            if (error) {
                console.warn('Fetch credits error:', error);
                els.creditCount.innerText = '3';
            } else {
                const credits = data?.credits ?? 0;
                els.creditCount.innerText = credits;
                // No more button mutation here. Logic moved to checkBtn click.
            }
        } catch (e) {
            console.error('Fetch credits exception:', e);
        }
    }

    // 4. Event Bindings
    if (els.navLoginBtn) {
        els.navLoginBtn.addEventListener('click', () => {
            document.getElementById('auth-section').scrollIntoView({ behavior: 'smooth' });
        });
    }

    if (els.copyBtn) {
        els.copyBtn.addEventListener('click', () => {
            const text = els.safeCopyText.innerText;
            if (text) {
                navigator.clipboard.writeText(text).then(() => {
                    const original = els.copyBtn.innerText;
                    els.copyBtn.innerText = 'Copied!';
                    setTimeout(() => els.copyBtn.innerText = original, 2000);
                });
            }
        });
    }

    // Get More Button Logic
    if (els.getMoreBtn) {
        els.getMoreBtn.addEventListener('click', () => {
            const isExt = document.body.classList.contains('is-extension');
            const url = 'https://www.listingguardai.com/pricing.html';
            
            if (isExt) {
                chrome.tabs.create({ url });
            } else {
                window.open(url, '_blank');
            }
        });
    }

    // Auto Grab Button
    if (els.autoFillBtn) {
        els.autoFillBtn.addEventListener('click', async () => {
            try {
                const [tab] = await chrome.tabs.query({ active: true, currentWindow: true });
                if (!tab || !tab.id) {
                    console.error('Tab check failed: tab or tab.id is missing');
                    alert('Unable to identify current tab.');
                    return;
                }

                // Inject Content Script if needed
                try {
                    await chrome.scripting.executeScript({
                        target: { tabId: tab.id },
                        files: ['content.js']
                    });
                } catch (injectionError) {
                    console.warn('Script injection warning:', injectionError);
                }

                chrome.tabs.sendMessage(tab.id, { action: 'extract_text' }, (response) => {
                    if (chrome.runtime.lastError) {
                        console.warn(chrome.runtime.lastError);
                        alert('Extract failed: Ensure you are on a valid page or allow file URL access.');
                        return;
                    }

                    if (response && response.success && response.data) {
                        const d = response.data;
                        let formattedText = '';

                        // Format Data in English
                        if (d.title || d.bullets.length > 0 || d.description) {
                            if (d.title) {
                                formattedText += `[Title]:\n${d.title}\n\n`;
                            }
                            
                            if (d.bullets.length > 0) {
                                formattedText += `[Bullet Points]:\n`;
                                d.bullets.forEach((b, i) => {
                                    formattedText += `${i+1}. ${b}\n`;
                                });
                                formattedText += `\n`;
                            }
                            
                            if (d.description) {
                                formattedText += `[Description]:\n${d.description}\n`;
                            }
                        } 
                        else if (d.fallback && d.fallback.length > 0) {
                            formattedText += `[Page Content]:\n`;
                            d.fallback.forEach((text, i) => {
                                formattedText += `--- Segment ${i+1} ---\n${text}\n\n`;
                            });
                        }

                        els.productInput.value = formattedText.trim();
                        els.checkBtn.click();
                    } else {
                        console.log('No valid content found.');
                    }
                });
            } catch (e) {
                console.error("Tab query failed:", e.message);
            }
        });
    }

    async function handleAuth(type) {
        const email = els.emailInput.value.trim();
        const password = els.pwdInput.value.trim();
        els.authMsg.innerText = '';

        if (!email || !password) {
            els.authMsg.innerText = 'Please enter email and password';
            return;
        }

        try {
            let error;
            if (type === 'login') {
                const res = await supabaseClient.auth.signInWithPassword({ email, password });
                error = res.error;
            } else {
                const res = await supabaseClient.auth.signUp({ email, password });
                error = res.error;
                if (!error && !res.data.session) {
                    els.authMsg.innerText = 'Sign up successful! Please verify your email.';
                    return;
                }
            }

            if (error) {
                els.authMsg.innerText = error.message;
            }
        } catch (e) {
            els.authMsg.innerText = e.message;
        }
    }

    if (els.loginSubmitBtn) {
        els.loginSubmitBtn.addEventListener('click', () => handleAuth('login'));
    }
    if (els.signupSubmitBtn) {
        els.signupSubmitBtn.addEventListener('click', () => handleAuth('signup'));
    }

    if (els.logoutBtn) {
        els.logoutBtn.addEventListener('click', async () => {
            if (!supabaseClient) return;
            try {
                await supabaseClient.auth.signOut();
                
                // 1. Force Clear Storage
                localStorage.clear();
                sessionStorage.clear();

                // 2. Manual UI Reset (Immediate)
                if (els.appView) els.appView.classList.add('hidden');
                if (els.landingView) els.landingView.classList.remove('hidden');
                if (els.navPrivate) els.navPrivate.classList.add('hidden');
                if (els.navPublic) els.navPublic.classList.remove('hidden');
                
                // 3. Reset Variables
                if (els.creditCount) els.creditCount.innerText = '-';
                if (els.productInput) els.productInput.value = '';
                
                // 4. Force state change handler
                await handleAuthStateChange(null);
                
                // Optional: Reload only if necessary, but manual reset is smoother
                // window.location.reload(); 

            } catch (e) {
                console.error('Logout failed:', e);
                localStorage.clear();
                sessionStorage.clear();
                window.location.reload();
            }
        });
    }

    // 5. Check Logic
    if (els.checkBtn) {
        els.checkBtn.addEventListener('click', async () => {
            const text = els.productInput.value.trim();
            const platform = els.platformSelect ? els.platformSelect.value : 'Amazon';

            if (!text) {
                alert('Please enter product text to analyze');
                return;
            }

            // Check Credits First
            const currentCredits = parseInt(els.creditCount.innerText) || 0;
            if (currentCredits <= 0) {
                if (els.creditsModal) {
                    els.creditsModal.classList.remove('hidden');
                } else {
                    alert('Beta credits exhausted! Please upgrade.');
                }
                return;
            }

            setLoading(true);

            try {
                const { data: { session }, error: sessionError } = await supabaseClient.auth.getSession();
                
                if (sessionError || !session) {
                    throw new Error('Session expired, please login again');
                }

                const token = session.access_token;
                console.log('Current Token:', token);

                const response = await fetch('https://www.listingguardai.com/api/check', {
                    method: 'POST',
                    headers: {
                        'Content-Type': 'application/json',
                        'Authorization': `Bearer ${token}`
                    },
                    body: JSON.stringify({ text, platform })
                });

                const result = await response.json();

                if (!response.ok) {
                    throw new Error(result.error || `Request failed (${response.status})`);
                }

                renderResult(result);
                
                if (typeof result.credits === 'number') {
                    els.creditCount.innerText = result.credits;
                } else {
                    fetchCredits(session.user.id);
                }

            } catch (e) {
                console.error('Check failed:', e);
                alert(`Analysis failed: ${e.message}`);
                setLoading(false, true);
            }
        });
    }

    // --- Helpers ---

    function setLoading(isLoading, reset = false) {
        if (reset) {
            els.resultPlaceholder.classList.remove('hidden');
            els.resultLoading.classList.add('hidden');
            els.resultContent.classList.add('hidden');
            els.checkBtn.disabled = false;
            els.checkBtn.innerText = 'Analyze Now';
            return;
        }

        if (isLoading) {
            els.resultPlaceholder.classList.add('hidden');
            els.resultLoading.classList.remove('hidden');
            els.resultContent.classList.add('hidden');
            els.checkBtn.disabled = true;
            els.checkBtn.innerText = 'AI Analyzing...';
            
            let p = 0;
            const interval = setInterval(() => {
                if (p >= 90) clearInterval(interval);
                else {
                    p += Math.random() * 5;
                    if (p > 90) p = 90;
                    updateProgress(p);
                }
            }, 200);
            window.currentLoadingInterval = interval;
        } else {
            if (window.currentLoadingInterval) clearInterval(window.currentLoadingInterval);
            updateProgress(100);
            setTimeout(() => {
                els.resultLoading.classList.add('hidden');
                els.resultContent.classList.remove('hidden');
                els.checkBtn.disabled = false;
                els.checkBtn.innerText = 'Analyze Now';
            }, 500);
        }
    }

    function updateProgress(percent) {
        const p = Math.round(percent);
        els.loadingBar.style.width = `${p}%`;
        els.loadingPercent.innerText = `${p}%`;
        
        if (p < 30) els.loadingStep.innerText = 'Connecting to Global Policy Database...';
        else if (p < 70) els.loadingStep.innerText = 'Analyzing Prohibited Terms...';
        else els.loadingStep.innerText = 'Generating Optimization Suggestions...';
    }

    function renderResult(data) {
        const { score, summary, details, safe_copy } = data;

        els.scoreVal.innerText = score;
        let badgeClass = '';
        let badgeText = 'Unknown';
        
        if (score >= 90) { badgeClass = 'safe'; badgeText = 'EXCELLENT'; }
        else if (score >= 70) { badgeClass = 'low'; badgeText = 'GOOD'; }
        else if (score >= 50) { badgeClass = 'medium'; badgeText = 'NEEDS WORK'; }
        else { badgeClass = 'high'; badgeText = 'HIGH RISK'; }
        
        els.scoreBadge.className = `score-badge ${badgeClass}`;
        els.scoreBadge.innerText = badgeText;

        els.summaryText.innerText = summary;

        els.detailsList.innerHTML = '';
        if (details && details.length > 0) {
            details.forEach(item => {
                // Map Level to Tag Class & Label
                let tagClass = 'low';
                let tagLabel = 'SUGGESTION';
                const lvl = (item.level || '').toLowerCase();
                
                if (lvl.includes('high') || lvl.includes('critical')) {
                    tagClass = 'CRITICAL';
                    tagLabel = 'CRITICAL';
                } else if (lvl.includes('medium') || lvl.includes('warning')) {
                    tagClass = 'WARNING';
                    tagLabel = 'WARNING';
                } else {
                    tagClass = 'SUGGESTION';
                    tagLabel = 'SUGGESTION';
                }
                
                const html = `
                    <div class="detail-item">
                        <div style="margin-bottom:8px;">
                            <span class="risk-tag ${tagClass}">${tagLabel}</span>
                            <span style="font-size:12px; font-weight:700; color:#0f172a;">${item.violation || 'General Issue'}</span>
                        </div>
                        
                        <div class="detail-row">
                            <span class="detail-label">Issue Found:</span>
                            <span class="detail-content">${item.reason}</span>
                        </div>
                        
                        ${item.suggestion ? `
                        <div class="detail-row">
                            <span class="detail-label">Mitigation:</span>
                            <span class="detail-content" style="color:#4f46e5;">${item.suggestion}</span>
                        </div>` : ''}
                    </div>
                `;
                els.detailsList.insertAdjacentHTML('beforeend', html);
            });
        } else {
            els.detailsList.innerHTML = '<div style="text-align:center; color:#94a3b8; font-size:12px;">🎉 No significant risks found</div>';
        }

        els.safeCopyText.innerText = safe_copy || 'No suggestions available.';
        setLoading(false);
    }

    // Init
    init();
});
